<?php
/*
Plugin Name: Simple Chat (Shortcodes)
Description: Chat simple entre usuarios con shortcodes [chat from="X" to="y"], [chatlist user="x"] y [chatadmin]. Crea tabla y carpeta de adjuntos al activarse.
Version: 1.0.0
Author: AlexCD2000
*/

// ====================================================================================
// ACTIVACIÓN: crea tabla y carpeta /chat-attachments en la raíz del sitio.
// ====================================================================================
register_activation_hook(__FILE__, function () {
    global $wpdb;

    // Nombre de tabla con prefijo WP
    $table = $wpdb->prefix . "simple_chats";
    $charset = $wpdb->get_charset_collate();

    // Estructura de almacenamiento: mensajes por "chat_key" (from-to ordenado)
    $sql = "CREATE TABLE IF NOT EXISTS `$table` (
        `id` bigint(20) unsigned NOT NULL AUTO_INCREMENT,
        `chat_key` varchar(190) NOT NULL,
        `post_id` bigint(20) unsigned NULL DEFAULT 0,
        `sender_id` bigint(20) unsigned NOT NULL,
        `receiver_id` bigint(20) unsigned NOT NULL,
        `message` longtext NULL,
        `attachment` varchar(255) NULL,
        `created_at` datetime NOT NULL,
        PRIMARY KEY (`id`),
        KEY `chat_key_idx` (`chat_key`),
        KEY `created_at_idx` (`created_at`)
    ) $charset;";

    require_once ABSPATH . "wp-admin/includes/upgrade.php";
    dbDelta($sql);

    // Tabla de reportes por chat_key
    $table_reports = $wpdb->prefix . "simple_chats_reports";
    $sql_reports = "CREATE TABLE IF NOT EXISTS `$table_reports` (
        `id` bigint(20) unsigned NOT NULL AUTO_INCREMENT,
        `chat_key` varchar(190) NOT NULL,
        `post_id` bigint(20) unsigned NULL DEFAULT 0,
        `reporter_id` bigint(20) unsigned NOT NULL,
        `reported_at` datetime NOT NULL,
        PRIMARY KEY (`id`),
        UNIQUE KEY `chat_key_unique` (`chat_key`)
    ) $charset;";
    $wpdb->query($sql_reports);
    // Migrar índice único para que sea por (chat_key, post_id)
    $idx_exists = $wpdb->get_var($wpdb->prepare("
        SELECT COUNT(1)
        FROM INFORMATION_SCHEMA.STATISTICS
        WHERE TABLE_SCHEMA = DATABASE()
          AND TABLE_NAME = %s
          AND INDEX_NAME = 'chat_post_unique'
    ", $table_reports));

    if (intval($idx_exists) == 0) {
        // Quitar único viejo por chat_key si existe
        $old_exists = $wpdb->get_var($wpdb->prepare("
            SELECT COUNT(1)
            FROM INFORMATION_SCHEMA.STATISTICS
            WHERE TABLE_SCHEMA = DATABASE()
              AND TABLE_NAME = %s
              AND INDEX_NAME = 'chat_key_unique'
        ", $table_reports));
        if (intval($old_exists) > 0) {
            $wpdb->query("ALTER TABLE `$table_reports` DROP INDEX `chat_key_unique`");
        }
        // Crear el nuevo único compuesto
        $wpdb->query("ALTER TABLE `$table_reports` ADD UNIQUE KEY `chat_post_unique` (`chat_key`,`post_id`)");
    }
    
    // Crear carpeta de adjuntos en la raíz del sitio: /chat-attachments
    $dir = ABSPATH . "chat-attachments";
    if (!file_exists($dir)) {
        wp_mkdir_p($dir);
    }
    
    // Establecer permisos de la carpeta a 0777
    @chmod($dir, 0777);
    //crear/actualizar .htaccess para forzar descarga y permitir acceso
    $htaccess_path = $dir . "/.htaccess";
    $ht = ""
    . "Options -Indexes\n"
    . "<IfModule mod_authz_core.c>\n"
    . "    Require all granted\n"
    . "</IfModule>\n"
    . "<IfModule mod_mime.c>\n"
    . "    AddType image/png .png\n"
    . "    AddType image/jpeg .jpg .jpeg\n"
    . "    AddType image/gif .gif\n"
    . "    AddType application/pdf .pdf\n"
    . "    AddType application/zip .zip\n"
    . "    AddDefaultCharset off\n"
    . "</IfModule>\n"
    . "<IfModule mod_headers.c>\n"
    . "    <FilesMatch \"\.(png|jpe?g|gif|pdf|zip)$\">\n"
    . "        Header set Content-Disposition \"attachment\"\n"
    . "    </FilesMatch>\n"
    . "</IfModule>\n";
    @file_put_contents($htaccess_path, $ht);
    
});

// ====================================================================================
// UTILIDADES: normalizar chat_key para un par de usuarios (from-to ordenado)
// ====================================================================================
function soc_normalize_chat_key($a, $b) {
    $a = intval($a);
    $b = intval($b);
    if ($a <= $b) {
        return $a . "-" . $b;
    } else {
        return $b . "-" . $a;
    }
}

// ====================================================================================
// ENQUEUE: jQuery (CDN) + definir window.ajaxurl global para el frontend
// ====================================================================================
add_action("wp_enqueue_scripts", function () {
    // CHANGE: encolamos jQuery desde CDN
    wp_enqueue_script(
        "soc-jquery-cdn",
        "https://code.jquery.com/jquery-latest.min.js",
        array(),
        null,
        true
    );

    // CHANGE: definimos window.ajaxurl para que exista en el frontend (logueados y visitantes)
    wp_add_inline_script(
        "soc-jquery-cdn",
        'window.ajaxurl = "' . esc_url(admin_url("admin-ajax.php")) . '";',
        "after"
    );
});

// ====================================================================================
// SHORTCODES
// ====================================================================================
include_once "shortcode-chat.php";
include_once "shortcode-report.php"; 
include_once "shortcode-chatlist.php";
include_once "shortcode-chatadmin.php";

// ====================================================================================
// AJAX – listar hilos por usuario con su último "last_at"
// ====================================================================================
add_action("wp_ajax_soc_list_threads", "soc_list_threads");
add_action("wp_ajax_nopriv_soc_list_threads", "soc_list_threads");
function soc_list_threads() {
    if (!is_user_logged_in()) { wp_send_json(array("success"=>false)); }

    $uid = isset($_POST["uid"]) ? intval($_POST["uid"]) : 0;
    if ($uid <= 0) { wp_send_json(array("success"=>false)); }

    global $wpdb;
    $table = $wpdb->prefix . "simple_chats";

    // agrupar por chat_key y ordenar por último mensaje
    $sql = "
        SELECT 
            chat_key,
            MAX(created_at) AS last_at,
            COUNT(*) AS total
        FROM `$table`
        WHERE chat_key LIKE %s OR chat_key LIKE %s
        GROUP BY chat_key
        ORDER BY last_at DESC
    ";
    $rows = $wpdb->get_results($wpdb->prepare($sql, $uid . "-%", "%-" . $uid), ARRAY_A);

    $out = array();
    if ($rows) {
        foreach ($rows as $r) {
            list($a, $b) = array_map("intval", explode("-", $r["chat_key"]));
            $other = ($a == $uid) ? $b : $a;
            $u = get_user_by("id", $other);
            $other_name = $u ? $u->display_name : ("User " . $other);
            $out[] = array(
                "chat_key"   => $r["chat_key"],
                "last_at"    => $r["last_at"],
                "other_id"   => $other,
                "other_name" => $other_name,
                "total"      => intval($r["total"])
            );
        }
    }

    wp_send_json(array("success"=>true, "data"=>$out));
}

// ====================================================================================
// AJAX – hilos como reportados
// ====================================================================================
// Devuelve true si el hilo (chat_key) está reportado
function soc_is_reported($chat_key, $post_id) {
    global $wpdb;
    $table_reports = $wpdb->prefix . "simple_chats_reports";
    $post_id = intval($post_id); // 0 si es general
    $exists = $wpdb->get_var($wpdb->prepare(
        "SELECT COUNT(1) FROM `$table_reports` WHERE `chat_key` = %s AND `post_id` = %d",
        $chat_key, $post_id
    ));
    if ($exists && intval($exists) > 0) {
        return true;
    }
    return false;
}

// AJAX para marcar un chat como reportado
add_action("wp_ajax_soc_report_chat", "soc_report_chat");
add_action("wp_ajax_nopriv_soc_report_chat", "soc_report_chat");
function soc_report_chat() {
    if (!is_user_logged_in()) { wp_send_json(array("success"=>false)); }

    $from    = isset($_POST["from"]) ? intval($_POST["from"]) : 0;
    $to      = isset($_POST["to"]) ? intval($_POST["to"]) : 0;
    $post_id = isset($_POST["post_id"]) ? intval($_POST["post_id"]) : 0;

    if ($from <= 0 || $to <= 0) { wp_send_json(array("success"=>false)); }

    // Normalizamos siempre la misma clave
    $chat_key = soc_normalize_chat_key($from, $to);

    global $wpdb;
    $table_reports = $wpdb->prefix . "simple_chats_reports";

    // Insertar o "upsert" sencillo: si ya existe, no duplicar
    $already = $wpdb->get_var($wpdb->prepare(
        "SELECT id FROM `$table_reports` WHERE chat_key=%s AND post_id=%d",
        $chat_key, $post_id
    ));    
    if (!$already) {
        $wpdb->insert($table_reports, array(
            "chat_key"    => $chat_key,
            "post_id"     => $post_id,
            "reporter_id" => get_current_user_id(),
            "reported_at" => current_time("mysql")
        ));
        
        // También guardar un mensaje visible en el hilo con la marca de hora
        $table_chats = $wpdb->prefix . "simple_chats";
        $report_ts = current_time("mysql");
        $system_msg = "Listing / User Reported, Investigation in progress - " . $report_ts;
        $wpdb->insert($table_chats, array(
            "chat_key"    => $chat_key,
            "post_id"     => $post_id,
            "sender_id"   => $from,   // el que reporta
            "receiver_id" => $to,     // el otro usuario del hilo
            "message"     => $system_msg,
            "attachment"  => "",      // sin adjunto
            "created_at"  => $report_ts
        ));
    }

    wp_send_json(array("success"=>true));
}

// ====================================================================================
// CSS Styles
// ====================================================================================
include_once "styles.php";